<?php
/**
 * Abstract Payment Provider
 *
 * @package AllPays
 * @since 1.0.0
 * @author AllPays
 * @copyright 2025 AllPays
 * @license GPL-2.0-or-later
 * @link https://allpays.co
 * @see https://allpays.co
 */

namespace AllPays\Abstracts;

defined( 'ABSPATH' ) || exit;

use WC_Order;
use AllPays\Core\API;
/**
 * Payment Provider Class
 * Provides base functionality for payment providers
 */
class Payment_Provider {

	/**
	 * Provider ID
	 *
	 * @var string
	 */
	public $id;

	/**
	 * Provider name
	 *
	 * @var string
	 */
	public $name;

	/**
	 * Provider description
	 *
	 * @var string
	 */
	public $description;

	/**
	 * Provider logo
	 *
	 * @var string
	 */
	public $logo;

	/**
	 * Provider notes
	 *
	 * @var string
	 */
	public $notes;

	/**
	 * Payment methods supported by this provider
	 *
	 * @var array
	 */
	public $supported_payment_methods = array();

	/**
	 * Initialize provider
	 *
	 * @param string $id Provider ID.
	 * @param string $name Provider name.
	 * @param string $description Provider description.
	 * @param string $notes Provider notes.
	 * @param string $logo Provider logo.
	 * @return void
	 */
	public function __construct( $id, $name, $description, $notes, $logo ) {
		$this->id          = empty( $id ) ? sanitize_title( $name ) : $id;
		$this->name        = $name;
		$this->description = $description;
		$this->notes       = $notes;
		$this->logo        = $logo;
	}

	/**
	 * Get the payment link
	 *
	 * @param WC_Order $order Order instance.
	 * @return string
	 */
	public function get_payment_link( WC_Order $order ): string {

		$encrypted_address = $order->get_meta( 'allpaysco_encrypted_address' );
		$polygon_address   = $order->get_meta( 'allpaysco_temporary_polygon_address' );

		// If no tracking address or polygon address exists, generate new one.
		if ( ! $encrypted_address || ! $polygon_address ) {
			$wallet_addresses  = API::generate_temporary_wallet_address( $order );
			$encrypted_address = $wallet_addresses['encrypted_address'];
			$polygon_address   = $wallet_addresses['polygon_address'];
		}

		$params = array(
			'encrypted_address=' . $encrypted_address,
			'amount=' . $order->get_total(),
			'provider=' . $this->id,
			'email=' . ( $order->get_user() !== false ? $order->get_user()->user_email : $order->get_billing_email() ),
			'currency=' . $order->get_currency(),
		);

		return ALLPAYSCO_PAYMENT_URL . '/pay?' . implode( '&', $params );
	}

	/**
	 * Get the redirect URL for payment processing
	 *
	 * @param WC_Order $order Order instance.
	 * @return string
	 */
	public function get_redirect_url( WC_Order $order ): string {
		return $order->get_checkout_payment_url( true );
	}
}
