<?php
/**
 * Base Payment Provider Gateway
 *
 * @package AllPays\PaymentMethods
 */

namespace AllPays\Payment_Methods;

defined( 'ABSPATH' ) || exit;

use AllPays\Abstracts\Abstract_Payment_Gateway;
use WC_Order;
use AllPays\Core\API;
use AllPays\Core\Registry;

/**
 * Base Payment Provider Gateway
 *
 * @package AllPays\PaymentMethods
 */
class Base_Payment_Provider_Gateway extends Abstract_Payment_Gateway {

	/**
	 * Provider Notes.
	 *
	 * @var string
	 */
	private $notes;

	/**
	 * Constructor
	 *
	 * @param string $id ID.
	 * @param string $method_name Method name.
	 * @param string $description Description.
	 * @param string $icon Icon.
	 * @param string $notes Notes.
	 */
	public function __construct( $id, $method_name, $description, $icon, $notes ) {

		$this->id          = $id;
		$this->method_name = $method_name;
		$this->description = $description;
		$this->icon        = $icon;
		$this->notes       = $notes;

		parent::__construct();

		/* translators: %s: Payment method name */
		$this->order_button_text = sprintf( __( 'Pay with %s', 'allpaysco-payment-gateway-for-woocommerce' ), $this->method_name );
	}

	/**
	 * Process payment.
	 *
	 * @param int $order_id Order ID.
	 * @return array
	 * @throws \Exception If payment fails.
	 */
	public function process_payment( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return array(
				'result'  => 'failure',
				'message' => __( 'Order not found.', 'allpaysco-payment-gateway-for-woocommerce' ),
			);
		}

		try {

			$available_providers = $this->get_available_providers( $order );

			if ( empty( $available_providers ) ) {
				/* translators: %s: Payment method name */
				throw new \Exception( sprintf( __( 'No payment providers available for %s.', 'allpaysco-payment-gateway-for-woocommerce' ), $this->method_name ) );
			}

			$order->set_payment_method( $this->id );

			// Mark as pending payment.
			$order->set_status( 'on-hold' );

			$order->save();

			// Empty cart.
			WC()->cart->empty_cart();

			return array(
				'result'   => 'success',
				'redirect' => $available_providers[0]->get_payment_link( $order ),
			);

		} catch ( \Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
			return array(
				'result'  => 'failure',
				'message' => $e->getMessage(),
			);
		}
	}

	/**
	 * Get available providers.
	 *
	 * @param WC_Order $order Order.
	 * @return array
	 */
	public function get_available_providers( WC_Order $order ) {
		$api_response        = API::get_available_gateways( $order, 'providers' );
		$available_providers = array();
		if ( ! empty( $api_response['payment_providers'] ) ) {
			foreach ( $api_response['payment_providers'] as $provider_data ) {
				if ( 'allpaysco_' . $provider_data['id'] === $this->id ) {
					$provider = Registry::get_provider( $provider_data['id'] );
					if ( $provider ) {
						$available_providers[] = $provider;
					}
					break;
				}
			}
		}

		return $available_providers;
	}

	/**
	 * Get notes.
	 *
	 * @return string
	 */
	public function get_notes() {
		return $this->notes;
	}
}
