<?php
/**
 * Base Payment Method Gateway
 *
 * @package AllPays\PaymentMethods
 */

namespace AllPays\Payment_Methods;

defined( 'ABSPATH' ) || exit;

use AllPays\Abstracts\Abstract_Payment_Gateway;
use WC_Order;
use AllPays\Core\API;
use AllPays\Core\Registry;

/**
 * Base Payment Method Gateway
 *
 * @package AllPays\PaymentMethods
 */
class Base_Payment_Method_Gateway extends Abstract_Payment_Gateway {

	/**
	 * Constructor
	 *
	 * @param string $id ID.
	 * @param string $method_name Method name.
	 * @param string $description Description.
	 * @param string $icon Icon.
	 */
	public function __construct( $id, $method_name, $description, $icon ) {

		$this->id          = $id;
		$this->method_name = $method_name;
		$this->description = $description;
		$this->icon        = $icon;

		parent::__construct();
	}

	/**
	 * Process payment.
	 *
	 * @param int $order_id Order ID.
	 * @return array
	 * @throws \Exception If payment fails.
	 */
	public function process_payment( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return array(
				'result'  => 'failure',
				'message' => __( 'Order not found.', 'allpaysco-payment-gateway-for-woocommerce' ),
			);
		}

		try {

			$order->set_payment_method( $this->id );

			// Mark as pending payment.
			$order->set_status( 'on-hold' );

			$order->save();

			// Empty cart.
			WC()->cart->empty_cart();

			return array(
				'result'   => 'success',
				'redirect' => $this->get_return_url( $order ),
			);

		} catch ( \Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
			return array(
				'result'  => 'failure',
				'message' => $e->getMessage(),
			);
		}
	}

	/**
	 * Get available providers.
	 *
	 * @param WC_Order $order Order.
	 * @return array
	 */
	public function get_available_providers( WC_Order $order ) {
		$api_response        = API::get_available_gateways( $order );
		$available_providers = array();
		if ( ! empty( $api_response['payment_methods'] ) ) {
			foreach ( $api_response['payment_methods'] as $method_data ) {
				if ( 'allpaysco_' . $method_data['id'] === $this->id && ! empty( $method_data['available_providers'] ) ) {
					foreach ( $method_data['available_providers'] as $provider_data ) {

						$providers_enabled = get_option( ALLPAYSCO_SETTINGS_ID . '_providers_enabled', array() );
						if ( ! isset( $providers_enabled[ $provider_data['id'] ] ) ) {
							continue;
						}

						$provider = Registry::get_provider( $provider_data['id'] );
						if ( $provider ) {
							$available_providers[] = $provider;
						}
					}
					break;
				}
			}
		}

		return $available_providers;
	}
}
