<?php
/**
 * Abstract Payment Method
 *
 * @package AllPays
 * @since 1.0.0
 * @author AllPays
 * @copyright 2025 AllPays
 * @license GPL-2.0-or-later
 * @link https://allpays.co
 * @see https://allpays.co
 */

namespace AllPays\Abstracts;

defined( 'ABSPATH' ) || exit;

use WC_Payment_Gateway;
use WC_Order;
use AllPays\Integrations\Base_Payment_Method_Integration;
use AllPays\Core\API;
/**
 * Abstract Payment Gateway Class
 * Provides base functionality for payment gateways
 */
abstract class Abstract_Payment_Gateway extends WC_Payment_Gateway {

	/**
	 * Payment gateway name
	 *
	 * @var string
	 */
	protected string $method_name;

	/**
	 * Providers supporting this payment gateway
	 *
	 * @var array
	 */

	/**
	 * Providers supporting this payment gateway
	 *
	 * @var array<string, Payment_Provider>
	 */
	protected array $providers = array();

	/**
	 * Init the instance
	 */
	public function __construct() {

		$this->id                 = 'allpaysco_' . ( $this->id ? $this->id : sanitize_title( $this->method_name ) );
		$this->has_fields         = false;
		$this->method_title       = sprintf( 'AllPays - %s', $this->method_name );
		$this->method_description = sprintf( 'Process payments over USDC using %s via AllPays.co', $this->method_name );

		// Load settings.
		$this->init_form_fields();
		$this->init_settings();

		// Get settings.
		$this->title   = $this->get_option( 'title', sprintf( '%s over Crypto', $this->method_name ) );
		$this->enabled = $this->get_option( 'enabled', 'no' );

		// Actions.
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_before_thankyou', array( $this, 'thankyou_page' ) );
		add_action( 'woocommerce_order_details_before_order_table', array( $this, 'display_order_payment_details' ), 10, 1 );

		// Remove default "Thank you" message for this payment method.
		add_filter( 'woocommerce_thankyou_order_received_text', array( $this, 'remove_thank_you_text' ), 10, 2 );

		// Customer emails.
		add_action( 'woocommerce_email_before_order_table', array( $this, 'email_instructions' ), 10, 3 );

		// Supports.
		$this->supports = array(
			'products',
		);

		// Add blocks support.
		if ( class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
			add_action(
				'woocommerce_blocks_payment_method_type_registration',
				array( $this, 'register_payment_method_block_support' )
			);
		}
	}

	/**
	 * Initialize form fields.
	 *
	 * @return void
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'enabled'     => array(
				'title'   => __( 'Enable/Disable', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'    => 'checkbox',
				/* translators: %s: Payment method title */
				'label'   => sprintf( __( 'Enable %s Payment', 'allpaysco-payment-gateway-for-woocommerce' ), $this->method_title ),
				'default' => 'no',
			),
			'title'       => array(
				'title'       => __( 'Title', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'        => 'text',
				'description' => __( 'This controls the title which the user sees during checkout.', 'allpaysco-payment-gateway-for-woocommerce' ),
				'default'     => $this->title,
				'desc_tip'    => true,
			),
			'description' => array(
				'title'       => __( 'Description', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'        => 'textarea',
				'description' => __( 'This controls the description which the user sees during checkout.', 'allpaysco-payment-gateway-for-woocommerce' ),
				'default'     => $this->description,
				'desc_tip'    => true,
			),
		);
	}

	/**
	 * Check if payment method is available.
	 *
	 * @return bool
	 */
	public function is_available() {

		if ( 'yes' !== $this->enabled ) {
			return false;
		}

		$polygon_wallet = get_option( ALLPAYSCO_SETTINGS_ID . '_polygon_wallet' );

		if ( empty( $polygon_wallet ) ) {
			return false;
		}

		if ( ! $this->providers ) {
			return false;
		}

		// Check cart and total.
		if ( ! WC()->cart || WC()->cart->get_total( 'edit' ) <= 0 ) {
			return false;
		}

		$order = new WC_Order();
		$order->set_billing_city( WC()->customer->get_billing_city() );
		$order->set_billing_state( WC()->customer->get_billing_state() );
		$order->set_billing_country( WC()->customer->get_billing_country() );
		$order->set_currency( get_woocommerce_currency() );
		$order->set_total( WC()->cart->get_total( 'edit' ) );
		$order->set_payment_method( $this->id );

		$available_providers = $this->get_available_providers( $order );

		if ( empty( $available_providers ) ) {
			return false;
		}

		return true;
	}



	/**
	 * Get payment links from available providers.
	 *
	 * @param WC_Order $order Order object.
	 * @return array Array of provider IDs and their payment links.
	 * @throws \Exception If payment links cannot be retrieved.
	 */
	protected function get_provider_payment_links( $order ) {
		$payment_links = array();

		$available_providers = $this->get_available_providers( $order );

		foreach ( $available_providers as $provider ) {
			try {
				$link = $provider->get_payment_link( $order, $this->id );
				if ( $link ) {
					$payment_links[ $provider->id ] = array(
						'name'  => $provider->name,
						'logo'  => $provider->logo,
						'link'  => $link,
						'notes' => $provider->notes,
					);
				}
			} catch ( \Exception $e ) {
				throw $e;
			}
		}

		return $payment_links;
	}

	/**
	 * Display payment details on order details page
	 *
	 * @param WC_Order $order Order object.
	 * @return void
	 */
	public function display_order_payment_details( $order ) {
		// Check if we're on the order details page and not the thank you page.
		if ( ! is_checkout() && is_wc_endpoint_url( 'view-order' ) ) {
			$this->thankyou_page( $order );
		}
	}

	/**
	 * Output for the order received page.
	 *
	 * @param int $order_id Order ID.
	 * @return void
	 */
	public function thankyou_page( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order || $this->id !== $order->get_payment_method() ) {
			return;
		}

		if ( $order->has_status( 'on-hold' ) ) {
			$this->enqueue_thankyou_styles();
			echo wp_kses_post( wpautop( wptexturize( $this->get_payment_instructions( $order ) ) ) );
		}
	}

	/**
	 * Add content to the WC emails.
	 *
	 * @param WC_Order $order Order object.
	 * @param bool     $sent_to_admin Sent to admin.
	 * @param bool     $plain_text Email format: plain text or HTML.
	 */
	public function email_instructions( $order, $sent_to_admin, $plain_text = false ) {

		if ( get_option( ALLPAYSCO_SETTINGS_ID . '_email_instructions', 'no' ) === 'no' ) {
			return;
		}

		if ( ! $sent_to_admin && $this->id === $order->get_payment_method() && $order->has_status( 'on-hold' ) ) {
			echo wp_kses_post( wpautop( wptexturize( $this->get_payment_instructions( $order ) ) ) . PHP_EOL );
		}
	}

	/**
	 * Get payment instructions.
	 *
	 * @param WC_Order $order Order object.
	 * @return string
	 */
	protected function get_payment_instructions( $order ) {
		$instructions = sprintf(
			'<h2 class="woocommerce-payment-instructions__title">%s</h2>',
			esc_html__( 'Payment Instructions', 'allpaysco-payment-gateway-for-woocommerce' )
		);

		$instructions .= sprintf(
			'<p class="woocommerce-payment-notice">%s</p>',
			sprintf(
					/* translators: %s: Payment method title */
				__( 'Your order #%1$s has been received and is awaiting payment via %2$s. Please follow the payment instructions below to complete your purchase.', 'allpaysco-payment-gateway-for-woocommerce' ),
				$order->get_order_number(),
				$this->get_title()
			)
		);

		$instructions_option = get_option( ALLPAYSCO_SETTINGS_ID . '_instructions', __( 'To complete your order, select one of the providers below. You will be redirected to the provider\'s platform to purchase crypto directly into our wallet. If you encounter any issues with one provider, you can try another. After the payment is confirmed, we will process your order automatically and send you an email with the order details.', 'allpaysco-payment-gateway-for-woocommerce' ) );

		if ( ! empty( $instructions_option ) ) {
			$instructions .= sprintf(
				'<p class="woocommerce-payment-notice">%s</p>',
				$instructions_option
			);
		}

		// Add payment links from providers if available.
		$payment_links = $this->get_provider_payment_links( $order );

		if ( ! empty( $payment_links ) ) {
			$instructions .= '<div class="allpays-providers-container">';
			$instructions .= '<div class="allpays-providers-header">';
			$instructions .= '<div class="allpays-header-provider">' . esc_html__( 'Payment Provider', 'allpaysco-payment-gateway-for-woocommerce' ) . '</div>';
			$instructions .= '</div>';

			foreach ( $payment_links as $provider_id => $provider_data ) {
				$instructions .= '<div class="allpays-provider-row">';
				// Left column - Logo.
				if ( ! empty( $provider_data['logo'] ) ) {
					$instructions .= sprintf(
						'<div class="allpays-provider-logo-wrapper"><img src="%1$s" alt="%2$s" class="allpays-provider-logo"/></div>',
						esc_url( $provider_data['logo'] ),
						esc_attr( $provider_data['name'] )
					);
				}
				// Right column - Content and Button.
				$instructions .= '<div class="allpays-provider-content">';
				$instructions .= '<div class="allpays-provider-info">';
				$instructions .= '<span class="allpays-provider-name">' . esc_html( $provider_data['name'] ) . '</span>';
				$instructions .= '<div class="allpays-provider-notes">';
				if ( ! empty( $provider_data['notes'] ) ) {
					$instructions .= wp_kses_post( $provider_data['notes'] );
				} else {
					$instructions .= sprintf(
						/* translators: %s: Provider name */
						esc_html__( 'Complete your payment securely with %s', 'allpaysco-payment-gateway-for-woocommerce' ),
						esc_html( $provider_data['name'] )
					);
				}
				$instructions .= '</div>';
				$instructions .= '</div>';
				$instructions .= sprintf(
					'<a href="%1$s" target="_blank" class="button allpays-provider-button">%2$s %3$s</a>',
					esc_url( $provider_data['link'] ),
					esc_html__( 'Pay with', 'allpaysco-payment-gateway-for-woocommerce' ),
					esc_html( $provider_data['name'] )
				);
				$instructions .= '</div>';
				$instructions .= '</div>';
			}

			$instructions .= '</div>';
		}

		return $instructions;
	}

	/**
	 * Get checkout payment fields HTML.
	 *
	 * @return string
	 */
	public function payment_fields_cancel() {
		$description = $this->get_description();
		$output      = '';

		if ( $description ) {
			$output = wpautop( wptexturize( $description ) );
		}

		$this->render_additional_payment_fields_cancel();
		return $output;
	}

	/**
	 * Render additional payment fields.
	 *
	 * @return void
	 */
	protected function render_additional_payment_fields_cancel() {
		// Implement in child classes if needed.
	}

	/**
	 * Add a provider to this payment method.
	 *
	 * @param Payment_Provider $provider Provider instance.
	 * @return void
	 */
	public function add_provider( Payment_Provider $provider ) {
		$this->providers[] = $provider;
	}

	/**
	 * Get available providers supporting this payment method.
	 *
	 * @param WC_Order $order Order object.
	 * @return Payment_Provider[]
	 */
	abstract public function get_available_providers( WC_Order $order );

	/**
	 * Register payment method block support
	 *
	 * @param \Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry Payment method registry.
	 * @return void
	 */
	public function register_payment_method_block_support( $payment_method_registry ) {
		$blocks_integration_class = new Base_Payment_Method_Integration( $this->id, get_class( $this ) );
		$payment_method_registry->register( $blocks_integration_class );
	}

	/**
	 * Remove the default thank you text for this payment method
	 *
	 * @param string   $text The default thank you text.
	 * @param WC_Order $order The order object.
	 * @return string
	 */
	public function remove_thank_you_text( $text, $order ) {
		if ( $order && $this->id === $order->get_payment_method() ) {
			return '';
		}
		return $text;
	}

	/**
	 * Enqueue styles for thank you page
	 *
	 * @return void
	 */
	public function enqueue_thankyou_styles() {
		wp_enqueue_style(
			'allpays-thankyou',
			plugins_url( '/assets/css/thankyou.css', dirname( __DIR__ ) ),
			array(),
			ALLPAYSCO_VERSION
		);
	}

	/**
	 * Get gateway icon.
	 *
	 * @return string
	 */
	public function get_icon() {
		if ( empty( $this->icon ) ) {
			return '';
		}

		$icon_html = '<img src="' . esc_url( $this->icon ) . '" alt="' . esc_attr( $this->get_title() ) . '" style="max-width:40px;height:auto;" />';

		return apply_filters( 'woocommerce_gateway_icon', $icon_html, $this->id );
	}
}
