<?php
/**
 * Base Payment Method Block Integration
 *
 * @package AllPays
 * @since 1.0.0
 * @author AllPays
 * @copyright 2025 AllPays
 * @license GPL-2.0-or-later
 * @link https://allpays.co
 * @see https://allpays.co
 */

namespace AllPays\Integrations;

defined( 'ABSPATH' ) || exit;

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

/**
 * Base class for payment method block integrations
 */
class Base_Payment_Method_Integration extends AbstractPaymentMethodType {

	/**
	 * The gateway instance.
	 *
	 * @var \WC_Payment_Gateway|\AllPays\Payment_Methods\Base_Payment_Method_Gateway|\AllPays\Payment_Methods\Base_Payment_Provider_Gateway
	 */
	protected $gateway;

	/**
	 * Gateway class name with namespace
	 *
	 * @var string
	 */
	protected $gateway_class;

	/**
	 * Constructor
	 *
	 * @param string $name Payment method name.
	 * @param string $gateway_class Gateway class name with namespace.
	 */
	public function __construct( string $name, string $gateway_class ) {
		$this->name          = $name;
		$this->gateway_class = $gateway_class;
	}

	/**
	 * Initializes the payment method type.
	 */
	public function initialize() {
		// Using autoloading instead of require.
		$this->settings = get_option( 'woocommerce_' . $this->get_name() . '_settings', array() );

		// Get the gateway instance.
		$gateways = \WC()->payment_gateways->payment_gateways();
		if ( isset( $gateways[ $this->name ] ) ) {
			$this->gateway = $gateways[ $this->name ];
		} else {
			$this->gateway = new $this->gateway_class();
		}
	}

	/**
	 * Returns if this payment method should be active.
	 *
	 * @return boolean
	 */
	public function is_active() {
		return true;
	}

	/**
	 * Returns an array of scripts/handles to be registered for this payment method.
	 *
	 * @return array
	 */
	public function get_payment_method_script_handles() {
		$handle = 'allpays-payment-method-blocks';

		// Register main script if not already registered.
		if ( ! wp_script_is( $handle, 'registered' ) ) {
			$script_path = 'assets/js/build/' . $handle . '.js';
			$asset_path  = ALLPAYSCO_PLUGIN_DIR . 'assets/js/build/' . $handle . '.asset.php';
			$script_url  = ALLPAYSCO_PLUGIN_URL . $script_path;

			$script_asset = array(
				'dependencies' => array( 'wp-element', 'wp-i18n', 'wc-blocks-registry', 'wc-settings' ),
				'version'      => ALLPAYSCO_VERSION,
			);

			if ( file_exists( $asset_path ) ) {
				$script_asset = require $asset_path;
			}

			wp_register_script(
				$handle,
				$script_url,
				$script_asset['dependencies'],
				$script_asset['version'],
				true
			);
		}

		// Check if inline script for this method already exists.
		$existing_data = wp_scripts()->get_data( $handle, 'after' );
		$method_script = sprintf(
			'registerAllPaysMethod({ id: "%s" });',
			$this->name,
			wp_json_encode( $this->get_supported_features() )
		);

		if ( ! is_array( $existing_data ) || ! in_array( $method_script, $existing_data, true ) ) {
			wp_add_inline_script( $handle, $method_script, 'after' );
		}

		return array( $handle );
	}

	/**
	 * Returns an array of key value pairs to be made available to the payment methods script.
	 *
	 * @return array
	 */
	public function get_payment_method_data() {
		$settings = array(
			'title'             => $this->gateway->get_title(),
			'description'       => $this->gateway->get_description(),
			'supports'          => $this->get_supported_features(),
			'icon'              => $this->gateway->get_icon(),
			'icon_url'          => $this->gateway->icon,
			'order_button_text' => $this->gateway->order_button_text,
			'notes'             => method_exists( $this->gateway, 'get_notes' ) ? $this->gateway->get_notes() : '',
			'testmode'          => 'yes' === $this->gateway->get_option( 'testmode', 'no' ),
		);

		$fee = get_option( ALLPAYSCO_SETTINGS_ID . '_fee' );
		if ( $fee ) {
			$is_discount   = strpos( $fee, '-' ) !== false;
			$is_percentage = strpos( $fee, '%' ) !== false;

			$fee = str_replace( array( '-', '%' ), '', $fee );

			if ( $is_percentage ) {
				$fee = '%' . $fee;
			} else {
				$fee = html_entity_decode( get_woocommerce_currency_symbol() ) . $fee;
			}

			if ( $is_discount ) {
				/* translators: %s: Off */
				$settings['badge'] = sprintf( __( '-%s OFF', 'allpaysco-payment-gateway-for-woocommerce' ), $fee );
			} else {
				/* translators: %s: Extra Fee */
				$settings['badge'] = sprintf( __( '%s Extra Fee', 'allpaysco-payment-gateway-for-woocommerce' ), $fee );
			}
		}

		return $settings;
	}

	/**
	 * Returns an array of supported features.
	 *
	 * @return string[]
	 */
	public function get_supported_features() {
		return is_array( $this->gateway->supports ) ? array_keys( array_filter( $this->gateway->supports ) ) : array( 'products' );
	}
}
