<?php
/**
 * AllPays Translations Handler
 *
 * @package AllPays
 */

namespace AllPays\Core;

defined( 'ABSPATH' ) || exit;

/**
 * Translations Class
 */
class Translations {

	/**
	 * Supported languages
	 *
	 * @var array
	 */
	private static $supported_languages = array(
		'de' => 'German',
		'it' => 'Italian',
		'es' => 'Spanish',
		'tr' => 'Turkish',
		'ru' => 'Russian',
		'fr' => 'French',
	);

	/**
	 * Initialize translations
	 *
	 * @return void
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'load_plugin_textdomain' ) );
		add_filter( 'plugin_locale', array( __CLASS__, 'set_locale' ), 10, 2 );
	}

	/**
	 * Load plugin text domain
	 *
	 * @return void
	 */
	public static function load_plugin_textdomain() {
		$locale = apply_filters( 'plugin_locale', get_locale(), 'allpaysco-payment-gateway-for-woocommerce' );
		$loaded = load_plugin_textdomain(
			'allpaysco-payment-gateway-for-woocommerce',
			false,
			dirname( ALLPAYSCO_PLUGIN_DIR ) . '/languages/'
		);

		if ( ! $loaded ) {
			// If no translation file is found, try to load from the languages directory.
			$mo_file = ALLPAYSCO_PLUGIN_DIR . 'languages/allpaysco-payment-gateway-for-woocommerce-' . $locale . '.mo';
			if ( file_exists( $mo_file ) ) {
				load_textdomain( 'allpaysco-payment-gateway-for-woocommerce', $mo_file );
			}
		}
	}

	/**
	 * Set locale for the plugin
	 *
	 * @param string $locale Current locale.
	 * @param string $domain Text domain.
	 * @return string
	 */
	public static function set_locale( $locale, $domain ) {
		if ( 'allpaysco-payment-gateway-for-woocommerce' === $domain ) {
			// Check if the current locale is supported.
			$locale_code = substr( $locale, 0, 2 );
			if ( isset( self::$supported_languages[ $locale_code ] ) ) {
				return $locale;
			}
		}
		return $locale;
	}

	/**
	 * Get supported languages
	 *
	 * @return array
	 */
	public static function get_supported_languages() {
		return self::$supported_languages;
	}

	/**
	 * Add a new supported language
	 *
	 * @param string $code Language code (e.g., 'fr' for French).
	 * @param string $name Language name.
	 * @return void
	 */
	public static function add_language( $code, $name ) {
		self::$supported_languages[ $code ] = $name;
	}

	/**
	 * Get language name by code
	 *
	 * @param string $code Language code.
	 * @return string|null
	 */
	public static function get_language_name( $code ) {
		return isset( self::$supported_languages[ $code ] ) ? self::$supported_languages[ $code ] : null;
	}

	/**
	 * Check if a language is supported
	 *
	 * @param string $code Language code.
	 * @return bool
	 */
	public static function is_language_supported( $code ) {
		return isset( self::$supported_languages[ $code ] );
	}
}
