<?php
/**
 * AllPays.co Settings Page
 *
 * @package AllPays
 */

namespace AllPays\Core;

use WC_Settings_Page;
use WC_Admin_Settings;

defined( 'ABSPATH' ) || exit;

/**
 * AllPays.Co Settings Class
 */
class Settings extends WC_Settings_Page {

	private const SETTINGS_TAB = 'checkout';

	/**
	 * Setting page tabs.
	 *
	 * @var array
	 */
	protected $tabs = array();

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->id = ALLPAYSCO_SETTINGS_ID;

		// Initialize tabs.
		$this->tabs = array(
			'general'   => __( 'General', 'allpaysco-payment-gateway-for-woocommerce' ),
			'providers' => __( 'Payment Providers', 'allpaysco-payment-gateway-for-woocommerce' ),
			'logs'      => __( 'Logs', 'allpaysco-payment-gateway-for-woocommerce' ),
		);

		add_filter( 'woocommerce_get_sections_' . self::SETTINGS_TAB, array( $this, 'add_settings_section' ) );
		add_filter( 'woocommerce_get_settings_' . self::SETTINGS_TAB, array( $this, 'add_settings' ), 10, 2 );
		add_action( 'woocommerce_admin_field_logs', array( $this, 'generate_logs_html' ) );

		// Add custom field types.
		add_action( 'woocommerce_admin_field_tabs_start', array( $this, 'generate_tabs_start_html' ) );
		add_action( 'woocommerce_admin_field_tab_content_start', array( $this, 'generate_tab_content_start_html' ) );
		add_action( 'woocommerce_admin_field_tab_content_end', array( $this, 'generate_tab_content_end_html' ) );

		add_filter( 'woocommerce_admin_settings_sanitize_option_' . ALLPAYSCO_SETTINGS_ID . '_polygon_wallet', array( $this, 'validate_polygon_wallet_address' ), 10, 3 );
		add_action( 'admin_notices', array( $this, 'check_required_settings' ) );
		add_filter( 'woocommerce_admin_settings_sanitize_option_' . ALLPAYSCO_SETTINGS_ID . '_fee', array( $this, 'validate_fee_field' ), 10, 3 );

		// Add tab display and scripts.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );

		// Add custom field types.
		add_action( 'woocommerce_admin_field_providers_list', array( $this, 'generate_providers_list_html' ) );
	}

	/**
	 * Enqueue admin scripts and styles
	 */
	public function enqueue_admin_scripts() {
		if ( isset( $_GET['section'] ) && ALLPAYSCO_SETTINGS_ID === $_GET['section'] ) {
			wp_enqueue_script(
				'allpays-admin-settings',
				plugins_url( 'assets/js/admin-settings.js', ALLPAYSCO_PLUGIN_FILE ),
				array( 'jquery' ),
				ALLPAYSCO_VERSION,
				true
			);
		}
	}

	/**
	 * Add settings section to WooCommerce
	 *
	 * @param array $sections Existing sections.
	 * @return array
	 */
	public function add_settings_section( $sections ) {
		$sections[ strtolower( $this->id ) ] = __( 'AllPays.co Settings', 'allpaysco-payment-gateway-for-woocommerce' );
		return $sections;
	}

	/**
	 * Generate tabs start HTML
	 *
	 * @param array $value Value.
	 */
	public function generate_tabs_start_html( $value ) {
		echo '<div class="allpays-settings-wrapper">';
		echo '<nav class="nav-tab-wrapper woo-nav-tab-wrapper">';
		foreach ( $this->tabs as $tab_id => $tab_name ) {
			printf(
				'<a href="#%s" class="nav-tab %s" data-tab="%s">%s</a>',
				esc_attr( $tab_id ),
				'general' === $tab_id ? 'nav-tab-active' : '',
				esc_attr( $tab_id ),
				esc_html( $tab_name )
			);
		}
		echo '</nav>';
	}

	/**
	 * Generate tab content start HTML
	 *
	 * @param array $value Value.
	 */
	public function generate_tab_content_start_html( $value ) {
		printf(
			'<div id="%s" class="allpays-tab-content" style="display: %s;">',
			esc_attr( $value['tab'] ),
			'general' === $value['tab'] ? 'block' : 'none'
		);
	}

	/**
	 * Generate tab content end HTML
	 *
	 * @param array $value Value.
	 */
	public function generate_tab_content_end_html( $value ) {
		echo '</div>';
		if ( isset( $value['is_last'] ) && $value['is_last'] ) {
			echo '</div>';
		}
	}

	/**
	 * Add settings
	 *
	 * @param array  $settings Existing settings.
	 * @param string $current_section Current section.
	 * @return array
	 */
	public function add_settings( $settings, $current_section ) {
		if ( ALLPAYSCO_SETTINGS_ID !== $current_section ) {
			return $settings;
		}

		return array(
			// Start tabs wrapper.
			array(
				'type' => 'tabs_start',
			),

			// General Tab Content.
			array(
				'type' => 'tab_content_start',
				'tab'  => 'general',
			),
			array(
				'id'    => ALLPAYSCO_SETTINGS_ID . '_general_section',
				'title' => __( 'AllPays General Settings', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'  => 'title',
				'desc'  => __( 'Configure AllPays payment gateway settings.', 'allpaysco-payment-gateway-for-woocommerce' ),
			),
			array(
				'id'    => ALLPAYSCO_SETTINGS_ID . '_enabled',
				'title' => __( 'Enable/Disable', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'  => 'checkbox',
				'desc'  => __( 'If disabled, the sub payment methods will be hidden in payment methods list.', 'allpaysco-payment-gateway-for-woocommerce' ),
			),
			array(
				'id'       => ALLPAYSCO_SETTINGS_ID . '_polygon_wallet',
				'title'    => __( 'POLYGON Wallet Address', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'     => 'text',
				'desc'     => __( 'Your POLYGON wallet address to receive USDC payments. <a href="https://allpays.co/blog/how-to-get-polygon-wallet" target="_blank">Learn how to get a Polygon wallet</a>', 'allpaysco-payment-gateway-for-woocommerce' ),
				'desc_tip' => __( 'Make sure to enter a valid POLYGON wallet address. Incorrect address may result in lost payments.', 'allpaysco-payment-gateway-for-woocommerce' ),
			),
			array(
				'id'      => ALLPAYSCO_SETTINGS_ID . '_mode',
				'title'   => __( 'Gateway Mode', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'    => 'select',
				'options' => array(
					'methods'   => __( 'Payment Methods', 'allpaysco-payment-gateway-for-woocommerce' ),
					'providers' => __( 'Payment Providers', 'allpaysco-payment-gateway-for-woocommerce' ),
				),
				'desc'    => __( 'Choose how customers will see payment options: "Methods" shows payment types (Credit Card, ApplePay, PayPal etc.), "Providers" shows payment companies (Stripe, Moonpay, Wert etc.). <br />Click here to <a href="https://allpays.co/support#woocommerce-gateway-mode" target="_blank">learn more about Gateway Mode</a>.', 'allpaysco-payment-gateway-for-woocommerce' ),
				'default' => 'methods',
			),
			array(
				'id'      => ALLPAYSCO_SETTINGS_ID . '_fee',
				'title'   => __( 'Add Fee/Discount', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'    => 'text',
				'desc'    => __( 'Add fee or discount. Use % for percentage (e.g. %5 or -%4) or number for fixed amount (e.g. 12 or -12). We automatically handle the currency symbol of your store. Maximum 99%.', 'allpaysco-payment-gateway-for-woocommerce' ),
				'default' => '',
			),
			array(
				'id'      => ALLPAYSCO_SETTINGS_ID . '_email_instructions',
				'title'   => __( 'Email Instructions', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'    => 'checkbox',
				'desc'    => __( 'Include payment provider list and instructions in new order emails', 'allpaysco-payment-gateway-for-woocommerce' ),
				'default' => 'yes',
			),
			array(
				'id'          => ALLPAYSCO_SETTINGS_ID . '_instructions',
				'title'       => __( 'Payment Instructions', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'        => 'textarea',
				'desc'        => __( 'Instructions that will be added to the thank you page and new order emails.', 'allpaysco-payment-gateway-for-woocommerce' ),
				'desc_tip'    => true,
				'placeholder' => __( 'Enter payment instructions here', 'allpaysco-payment-gateway-for-woocommerce' ),
				'css'         => 'width: 100%; min-height: 100px;',
				'default'     => __( 'To complete your order, select one of the providers below. You will be redirected to the provider\'s platform to purchase crypto directly into our wallet. If you encounter any issues with one provider, you can try another. After the payment is confirmed, we will process your order automatically and send you an email with the order details.', 'allpaysco-payment-gateway-for-woocommerce' ),
			),
			array(
				'id'   => ALLPAYSCO_SETTINGS_ID . '_fee_sectionend',
				'type' => 'sectionend',
			),
			array(
				'type' => 'tab_content_end',
			),

			// Providers Tab Content.
			array(
				'type' => 'tab_content_start',
				'tab'  => 'providers',
			),
			array(
				'id'    => ALLPAYSCO_SETTINGS_ID . '_providers_section',
				'title' => __( 'Payment Providers Settings', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'  => 'title',
				'desc'  => __( 'Enable or disable payment providers. These settings will be applied both when Gateway Mode is set to "Payment Methods" and "Payment Providers". <br /> <br /> <strong>Note:</strong> If a provider is disabled, it will not be available for selection in the WooCommerce\'s payment methods list when Gateway Mode is set to "Payment Providers". You can click here to <a href="https://allpays.co/support#woocommerce-gateway-mode" target="_blank">learn more about Gateway Mode and payment providers</a>.', 'allpaysco-payment-gateway-for-woocommerce' ),
			),
			array(
				'id'      => ALLPAYSCO_SETTINGS_ID . '_providers_enabled',
				'title'   => __( 'Available Providers', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'    => 'providers_list',
				'default' => array(),
			),
			array(
				'id'   => ALLPAYSCO_SETTINGS_ID . '_providers_sectionend',
				'type' => 'sectionend',
			),
			array(
				'type' => 'tab_content_end',
			),

			// Logs Tab Content.
			array(
				'type' => 'tab_content_start',
				'tab'  => 'logs',
			),
			array(
				'id'    => ALLPAYSCO_SETTINGS_ID . '_logging_section',
				'title' => __( 'Logging Settings', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'  => 'title',
			),
			array(
				'id'      => ALLPAYSCO_SETTINGS_ID . '_logging_enabled',
				'title'   => __( 'Enable Logging', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'    => 'checkbox',
				'desc'    => __( 'Enable transaction logging', 'allpaysco-payment-gateway-for-woocommerce' ),
				'default' => 'yes',
			),
			array(
				'id'      => ALLPAYSCO_SETTINGS_ID . '_debug_logging',
				'title'   => __( 'Debug Logs', 'allpaysco-payment-gateway-for-woocommerce' ),
				'type'    => 'checkbox',
				'desc'    => __( 'Include debug level logs', 'allpaysco-payment-gateway-for-woocommerce' ),
				'default' => 'no',
			),
			array(
				'id'   => ALLPAYSCO_SETTINGS_ID . '_logs',
				'type' => 'logs',
			),
			array(
				'id'   => ALLPAYSCO_SETTINGS_ID . '_logging_sectionend',
				'type' => 'sectionend',
			),
			array(
				'type'    => 'tab_content_end',
				'is_last' => true,
			),
		);
	}

	/**
	 * Generate Logs HTML
	 *
	 * @param array $value Value.
	 */
	public function generate_logs_html( array $value ): void {
		echo '<tr valign="top">';
		echo '<th scope="row" class="titledesc">';
		echo '<label>' . esc_html__( 'Transaction Logs', 'allpaysco-payment-gateway-for-woocommerce' ) . '</label>';
		echo '</th>';
		echo '<td class="forminp">';
		echo '<p>' . sprintf(
			/* translators: %s: WooCommerce status logs URL */
			esc_html__( 'Logs are maintained by WooCommerce\'s logging system and can be viewed in the %s.', 'allpaysco-payment-gateway-for-woocommerce' ),
			'<a href="' . esc_url( admin_url( 'admin.php?page=wc-status&tab=logs' ) ) . '">' .
			esc_html__( 'WooCommerce Status > Logs', 'allpaysco-payment-gateway-for-woocommerce' ) .
			'</a>'
		) . '</p>';
		echo '</td>';
		echo '</tr>';
	}

	/**
	 * Validate wallet address
	 *
	 * @param string $value Value.
	 * @param string $option Option.
	 * @param string $raw_value Raw value.
	 * @return string
	 */
	public function validate_polygon_wallet_address( $value, $option, $raw_value ) {
		$wallet_address = trim( $raw_value );
		$current_value  = get_option( $option['id'], '' );

		if ( empty( $wallet_address ) ) {
			WC_Admin_Settings::add_error( __( 'Wallet address cannot be empty.', 'allpaysco-payment-gateway-for-woocommerce' ) );
			return get_option( $option['id'], '' );
		}

		if ( substr( $wallet_address, 0, 2 ) !== '0x' ) {
			WC_Admin_Settings::add_error( __( 'Invalid wallet address format. Address must start with "0x"', 'allpaysco-payment-gateway-for-woocommerce' ) );
			return get_option( $option['id'], '' );
		}

		if ( strlen( $wallet_address ) !== 42 ) {
			WC_Admin_Settings::add_error( __( 'Invalid wallet address length. Address must be 42 characters long.', 'allpaysco-payment-gateway-for-woocommerce' ) );
			return get_option( $option['id'], '' );
		}

		if ( ! preg_match( '/^0x[0-9a-f]{40}$/i', $wallet_address ) ) {
			WC_Admin_Settings::add_error( __( 'Invalid wallet address format. Address must be hexadecimal.', 'allpaysco-payment-gateway-for-woocommerce' ) );
			return get_option( $option['id'], '' );
		}

		// Check if not USDC contract address.
		$polygon_usdc_address = 0x3c499c542cef5e3811e1192ce70d8cc03d5c3359;
		if ( strtolower( $wallet_address ) === strtolower( (string) $polygon_usdc_address ) ) {
			WC_Admin_Settings::add_error( __( 'Invalid wallet address: Cannot use USDC contract address', 'allpaysco-payment-gateway-for-woocommerce' ) );
			return get_option( $option['id'], '' );
		}

		return $wallet_address;
	}

	/**
	 * Check and display admin notices for required settings
	 */
	public function check_required_settings() {
		if ( ! is_admin() ) {
			return;
		}

		$settings_url = admin_url( 'admin.php?page=wc-settings&tab=' . self::SETTINGS_TAB . '&section=' . ALLPAYSCO_SETTINGS_ID );
		$settings     = self::retrieve_settings();

		// Check wallet address.
		if ( empty( $settings['polygon_wallet'] ) ) {
			?>
			<div class="notice notice-error">
				<p>
					<?php
					printf(
						/* translators: %s: Settings URL */
						esc_html__( 'AllPays.co: Please set your Polygon wallet address in the %s to receive payments.', 'allpaysco-payment-gateway-for-woocommerce' ),
						'<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'AllPays.co settings', 'allpaysco-payment-gateway-for-woocommerce' ) . '</a>'
					);
					?>
				</p>
			</div>
			<?php
		}

		// Check enabled providers.
		$enabled_providers = get_option( ALLPAYSCO_SETTINGS_ID . '_providers_enabled', array() );
		if ( empty( $enabled_providers ) ) {
			?>
			<div class="notice notice-error">
				<p>
					<?php
					printf(
						/* translators: %s: Settings URL */
						esc_html__( 'AllPays.co: No payment providers are enabled. Please enable at least one provider in the %s to start accepting payments.', 'allpaysco-payment-gateway-for-woocommerce' ),
						'<a href="' . esc_url( $settings_url . '#providers' ) . '">' . esc_html__( 'Payment Providers settings', 'allpaysco-payment-gateway-for-woocommerce' ) . '</a>'
					);
					?>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Get settings
	 *
	 * @return array
	 */
	public static function retrieve_settings() {
		return array(
			'enabled'                  => get_option( ALLPAYSCO_SETTINGS_ID . '_enabled' ),
			'api_key'                  => get_option( ALLPAYSCO_SETTINGS_ID . '_api_key' ),
			'polygon_wallet'           => get_option( ALLPAYSCO_SETTINGS_ID . '_polygon_wallet' ),
			'waiting_payment_status'   => get_option( ALLPAYSCO_SETTINGS_ID . '_waiting_payment_status' ),
			'payment_completed_status' => get_option( ALLPAYSCO_SETTINGS_ID . '_payment_completed_status' ),
			'fee'                      => get_option( ALLPAYSCO_SETTINGS_ID . '_fee' ),
		);
	}

	/**
	 * Validate fee/discount setting.
	 *
	 * @param string $value The sanitized value.
	 * @param array  $option The option array.
	 * @param string $raw_value The raw value from the form.
	 * @return string The validated value or existing value if validation fails.
	 */
	public function validate_fee_field( $value, $option, $raw_value ) {
		$current_value = get_option( $option['id'], '' );

		if ( empty( $raw_value ) ) {
			return '';
		}

		// Use raw_value to preserve % characters.
		$value = preg_replace( '/\s+/', '', $raw_value );

		// Check for zero value.
		$check_zero = str_replace( array( '%', '-' ), '', $value );
		// Validate format: optional minus, optional %, numbers with optional decimal, optional %.
		if ( ! preg_match( '/^(-)?%?\d*\.?\d+%?$/', $value ) || 0.0 === (float) $check_zero ) {
			WC_Admin_Settings::add_error(
				__( 'Invalid fee/discount format. Use %5 for 5% fee, -%5 for 5% discount, 10 for $10 fee, or -10 for $10 discount.', 'allpaysco-payment-gateway-for-woocommerce' )
			);
			return $current_value;
		}

		// Check percentage limit.
		if ( strpos( $value, '%' ) !== false ) {
			$check_value = str_replace( '%', '', $value );
			if ( abs( (float) $check_value ) > 99 ) {
				WC_Admin_Settings::add_error(
					__( 'Fee/Discount percentage cannot be more than 99%.', 'allpaysco-payment-gateway-for-woocommerce' )
				);
				return $current_value;
			}
		}

		return $value;
	}

	/**
	 * Generate Providers List HTML
	 *
	 * @param array $value Field value.
	 */
	public function generate_providers_list_html( array $value ): void {
		$gateways = API::get_gateways();

		if ( empty( $gateways ) || empty( $gateways['payment_providers'] ) ) {
			echo '<tr><td colspan="2">';
			echo '<p>' . esc_html__( 'No payment providers available.', 'allpaysco-payment-gateway-for-woocommerce' ) . '</p>';
			echo '</td></tr>';
			return;
		}

		// Start table wrapper.
		echo '<tr><td colspan="2" class="forminp forminp-providers_list">';
		echo '<table class="widefat" style="margin-top: 1em;">';

		// Table header.
		echo '<thead>';
		echo '<tr>';
		echo '<th class="check-column"></th>';
		echo '<th class="logo-column"></th>';
		echo '<th class="provider-details-column">' . esc_html__( 'Provider Details', 'allpaysco-payment-gateway-for-woocommerce' ) . '</th>';
		echo '</tr>';
		echo '</thead>';

		// Table body.
		echo '<tbody>';
		foreach ( $gateways['payment_providers'] as $provider ) {
			$provider_id     = sanitize_key( $provider['id'] );
			$providers_array = get_option( $value['id'], array() );

			echo '<tr>';

			// Checkbox column with WooCommerce switch styling.
			echo '<td class="check-column">';
			echo '<div class="woocommerce-input-toggle woocommerce-input-toggle--' .
				( isset( $providers_array[ $provider_id ] ) ? 'enabled' : 'disabled' ) . '">';
			echo '<input
				name="' . esc_attr( $value['id'] ) . '[' . esc_attr( $provider_id ) . ']"
				id="' . esc_attr( $value['id'] . '_' . $provider_id ) . '"
				type="checkbox"
				class="allpays-provider-checkbox"
				value="1"
				' . checked( isset( $providers_array[ $provider_id ] ), true, false ) . '
				style="display:none;"
			/>';
			echo '</div>';
			echo '</td>';

			// Logo column.
			echo '<td class="logo-column">';
			if ( ! empty( $provider['logo'] ) ) {
				/**
				 * Allow external API images in provider logos
				 *
				 * @param bool   $allow_external Always false
				 * @param string $url           The image URL
				 */
				add_filter(
					'wp_kses_allowed_html',
					function ( $allowed_html ) {
						$allowed_html['img'] = array(
							'src'     => array(),
							'alt'     => array(),
							'class'   => array(),
							'loading' => array(),
						);
						return $allowed_html;
					},
					10,
					1
				);

				printf(
					'<img src="%s" alt="%s" class="provider-logo" loading="lazy" />',
					esc_url( $provider['logo'] ),
					esc_attr( $provider['name'] )
				);
			}
			echo '</td>';

			// Provider details column.
			echo '<td class="provider-details-column">';
			echo '<h4 class="provider-name">' . esc_html( $provider['name'] ) . '</h4>';
			if ( ! empty( $provider['description'] ) ) {
				echo '<div class="provider-description">' . esc_html( $provider['description'] );

				echo '</div>';
			}
			if ( ! empty( $provider['notes'] ) ) {
				echo '<div class="provider-notes">' . esc_html( $provider['notes'] ) . '</div>';
			}

							// Add supported methods names right after name.
			if ( ! empty( $provider['supported_methods'] ) ) {
				echo '<span class="provider-supported-methods">Supported methods: ';
				echo esc_html( implode( ', ', array_column( $provider['supported_methods'], 'name' ) ) );
				echo '</span>';
			}

			echo '</td>';

			echo '</tr>';
		}
		echo '</tbody>';
		echo '</table>';
		echo '</td></tr>';

		// Add CSS inline for styling.
		?>
		<style>
			.forminp-providers_list .widefat {
				border: 1px solid #c3c4c7;
				box-shadow: 0 1px 1px rgba(0,0,0,.04);
				width: 100%;
			}
			.forminp-providers_list .check-column {
				width: auto !important;
				white-space: nowrap;
				padding: 8px !important;
				vertical-align: middle !important;
			}
			.forminp-providers_list .logo-column {
				width: auto !important;
				white-space: nowrap;
				text-align: center;
				vertical-align: middle;
				padding: 8px;
			}
			.forminp-providers_list .provider-details-column {
				width: 100% !important;
			}
			.provider-logo {
				width: 60px;
				height: auto;
				display: block;
				margin: 0 auto;
			}
			.provider-details {
				padding: 12px 10px;
			}
			.provider-name {
				margin: 0 0 8px;
				font-size: 14px;
				font-weight: 600;
			}
			.provider-description {
				margin-bottom: 6px;
			}
			.provider-notes {
				color: #666;
				font-style: italic;
				font-size: 13px;
			}
			.forminp-providers_list thead th {
				padding: 8px;
			}
			.check-column {
				cursor: pointer;
			}
			.woocommerce-input-toggle {
				margin: 0 auto;
			}
			.forminp-providers_list .widefat tr:nth-child(odd) {
				background-color: #f9f9f9;
			}
			.forminp-providers_list .widefat tr:hover {
				background-color: #f5f5f5;
			}
			.provider-supported-methods {
				color: #666;
				font-size: 12px;
				margin-top: 8px;
			}
		</style>
		<script>
		jQuery(document).ready(function($) {
			// Toggle switch when clicking the cell.
			$('.check-column').click(function(e) {
				if ($(e.target).is('input')) return;

				var checkbox = $(this).find('input[type="checkbox"]');
				var toggle = $(this).find('.woocommerce-input-toggle');

				// Toggle checkbox and trigger change
				checkbox.prop('checked', !checkbox.prop('checked')).trigger('change');

				// Update toggle class based on new checkbox state
				toggle.toggleClass('woocommerce-input-toggle--enabled woocommerce-input-toggle--disabled');

			});


		});
		</script>
		<?php
	}
}
