<?php
/**
 * Registry Class
 *
 * @package AllPays
 * @since 1.0.0
 * @author AllPays
 * @copyright 2025 AllPays
 * @license GPL-2.0-or-later
 * @link https://allpays.co
 * @see https://allpays.co
 */

namespace AllPays\Core;

use AllPays\Abstracts\Abstract_Payment_Gateway;
use AllPays\Abstracts\Payment_Provider;

defined( 'ABSPATH' ) || exit;

/**
 * Registry Class
 * Manages registration of payment gateways and providers
 */
class Registry {

	/**
	 * Registry of payment gateways
	 *
	 * @var array<string, Abstract_Payment_Gateway>
	 */
	private static array $payment_gateways = array();

	/**
	 * Registry of providers
	 *
	 * @var array<string, Payment_Provider>
	 */
	private static array $providers = array();

	/**
	 * Register a payment gateway
	 *
	 * @param Abstract_Payment_Gateway $gateway Payment gateway instance.
	 * @return void
	 */
	public static function register_payment_gateway( Abstract_Payment_Gateway $gateway ): void {
		self::$payment_gateways[ $gateway->id ] = $gateway;

		add_filter(
			'woocommerce_payment_gateways',
			function ( $gateways ) use ( $gateway ) {
				$gateways[] = $gateway;
				return $gateways;
			}
		);
	}

	/**
	 * Register a provider
	 *
	 * @param Payment_Provider $provider Provider instance.
	 * @return void
	 */
	public static function register_provider( Payment_Provider $provider ): void {
		self::$providers[ $provider->id ] = $provider;
	}

	/**
	 * Get all registered payment gateways
	 *
	 * @return array<string, Abstract_Payment_Gateway>
	 */
	public static function get_payment_gateways(): array {
		return self::$payment_gateways;
	}

	/**
	 * Get all registered providers
	 *
	 * @return array<string, Payment_Provider>
	 */
	public static function get_providers(): array {
		return self::$providers;
	}

	/**
	 * Get provider by ID
	 *
	 * @param string $provider_id Provider ID.
	 * @return Payment_Provider|null
	 */
	public static function get_provider( string $provider_id ): ?Payment_Provider {
		return self::$providers[ $provider_id ] ?? null;
	}

	/**
	 * Get payment gateways by provider ID
	 *
	 * @param string $provider_id Provider ID.
	 * @return array<string, Abstract_Payment_Gateway>
	 */
	public static function get_payment_gateways_by_provider_id( string $provider_id ): array {
		$gateways = array();
		foreach ( self::$payment_gateways as $gateway ) {
			if ( isset( $gateway->providers[ $provider_id ] ) ) {
				$gateways[ $gateway->id ] = $gateway;
			}
		}
		return $gateways;
	}
}
