<?php
/**
 * Logger Class
 *
 * @package AllPays\Core
 */

namespace AllPays\Core;

defined( 'ABSPATH' ) || exit;

/**
 * Logger Class
 * Handles all logging functionality
 */
class Logger {
	/**
	 * Log source/handle
	 */
	private const LOG_SOURCE = 'allpaysco';

	/**
	 * Logger instance
	 *
	 * @var \WC_Logger|null
	 */
	private static $logger = null;

	/**
	 * Get WC_Logger instance
	 *
	 * @return \WC_Logger
	 */
	private static function get_logger(): \WC_Logger {
		if ( null === self::$logger ) {
			self::$logger = wc_get_logger();
		}
		return self::$logger;
	}

	/**
	 * Log a debug message
	 *
	 * @param string $message The message to log.
	 * @param mixed  $data Additional data to log.
	 */
	public static function debug( string $message, $data = null ): void {
		self::write_log( $message, $data, 'debug' );
	}

	/**
	 * Log an info message
	 *
	 * @param string $message The message to log.
	 * @param mixed  $data Additional data to log.
	 */
	public static function info( string $message, $data = null ): void {
		self::write_log( $message, $data, 'info' );
	}

	/**
	 * Log an error message
	 *
	 * @param string $message The message to log.
	 * @param mixed  $data Additional data to log.
	 */
	public static function error( string $message, $data = null ): void {
		self::write_log( $message, $data, 'error' );
	}

	/**
	 * Write log message
	 *
	 * @param string $message The message to log.
	 * @param mixed  $data Additional data to log.
	 * @param string $level Log level.
	 */
	private static function write_log( string $message, $data = null, string $level = 'debug' ): void {
		if ( 'yes' !== get_option( ALLPAYSCO_SETTINGS_ID . '_logging_enabled', 'yes' ) ) {
			return;
		}

		// Skip debug logs if debug logging is not enabled.
		if ( 'debug' === $level && 'yes' !== get_option( ALLPAYSCO_SETTINGS_ID . '_debug_logging', 'no' ) ) {
			return;
		}

		self::get_logger()->log(
			$level,
			$message,
			array(
				'source' => self::LOG_SOURCE,
				'data'   => $data,
			)
		);
	}
}
