<?php
/**
 * Health Check Handler
 *
 * @package AllPays
 */

namespace AllPays\Core;

defined( 'ABSPATH' ) || exit;

/**
 * Health Check Class
 * Handles system health checks and API status monitoring
 */
class Health_Check {

	/**
	 * Check system and API health
	 *
	 * @return array{success: bool, messages: array<string>, data?: array}
	 */
	public static function check_health(): array {
		$result = array(
			'success'  => true,
			'messages' => array(),
		);

		// Check WordPress version requirement.
		if ( ! self::check_wp_version() ) {
			$result['success']    = false;
			$result['messages'][] = array(
				'message'     => self::format_message(
					sprintf(
					/* translators: %s: Minimum WordPress version */
						__( 'Requires WordPress version %s or higher.', 'allpaysco-payment-gateway-for-woocommerce' ),
						'5.6'
					)
				),
				'type'        => 'error',
				'dismissible' => false,
			);
		}

		// Check WooCommerce version requirement.
		if ( ! self::check_wc_version() ) {
			$result['success']    = false;
			$result['messages'][] = array(
				'message'     => self::format_message(
					sprintf(
					/* translators: %s: Minimum WooCommerce version */
						__( 'Requires WooCommerce version %s or higher.', 'allpaysco-payment-gateway-for-woocommerce' ),
						'6.0'
					)
				),
				'type'        => 'error',
				'dismissible' => false,
			);
		}

		// Check API health using API class.
		$api_response = API::get_health_status();

		if ( null === $api_response ) {
			$result['success']    = false;
			$result['messages'][] = array(
				'message'     => self::format_message(
					sprintf(
					/* translators: %s: Status page URL */
						__( 'Unable to connect to API. Payments temporarily disabled. Check <a href="https://api.allpays.co/status" target="_blank">https://api.allpays.co/status</a> for more information.', 'allpaysco-payment-gateway-for-woocommerce' )
					)
				),
				'type'        => 'error',
				'dismissible' => false,
			);
		} elseif ( empty( $api_response['success'] ) || empty( $api_response['data'] ) ) {
			$result['success']    = false;
			$result['messages'][] = array(
				'message'     => self::format_message(
					__( 'Invalid health check response format.', 'allpaysco-payment-gateway-for-woocommerce' )
				),
				'type'        => 'error',
				'dismissible' => false,
			);
		} else {
			// Check maintenance status.
			if ( 'MAINTENANCE' === $api_response['data']['status'] ) {
				$result['success']    = false;
				$result['messages'][] = array(
					'message'     => self::format_message(
						! empty( $api_response['data']['message'] )
							? $api_response['data']['message']
							: __( 'API is currently under maintenance.', 'allpaysco-payment-gateway-for-woocommerce' )
					),
					'type'        => 'error',
					'dismissible' => false,
				);
			}

			// Check WooCommerce plugin version compatibility.
			if ( ! empty( $api_response['data']['WC_PLUGIN_MIN_VERSION'] ) ) {
				$required_version = $api_response['data']['WC_PLUGIN_MIN_VERSION'];
				if ( version_compare( ALLPAYSCO_VERSION, $required_version, '<' ) ) {
					$result['success']    = false;
					$result['messages'][] = array(
						'message'     => self::format_message(
							sprintf(
							/* translators: %s: Required plugin version */
								__( 'Your plugin (v%1$s) is outdated. Please update to v%2$s or later to continue accepting payments through AllPays.co', 'allpaysco-payment-gateway-for-woocommerce' ),
								ALLPAYSCO_VERSION,
								$required_version
							)
						),
						'type'        => 'error',
						'dismissible' => false,
					);
				}
			}

			// Check for recommended version updates.
			if ( ! empty( $api_response['data']['WC_PLUGIN_VERSION'] ) ) {
				$recommended_version = $api_response['data']['WC_PLUGIN_VERSION'];
				if ( version_compare( ALLPAYSCO_VERSION, $recommended_version, '<' ) ) {
					$result['messages'][] = array(
						'message'     => self::format_message(
							sprintf(
							/* translators: %s: Recommended plugin version */
								__( 'A new version (v%s) of AllPays.co plugin is available. Please update when possible for the latest features and improvements.', 'allpaysco-payment-gateway-for-woocommerce' ),
								$recommended_version
							)
						),
						'type'        => 'warning',
						'dismissible' => true,
					);
				}
			}

			$result['data'] = $api_response['data'];
		}

		return $result;
	}

	/**
	 * Check WordPress version requirement
	 *
	 * @return bool
	 */
	private static function check_wp_version(): bool {
		return version_compare( get_bloginfo( 'version' ), '5.6', '>=' );
	}

	/**
	 * Check WooCommerce version requirement
	 *
	 * @return bool
	 */
	private static function check_wc_version(): bool {
		if ( ! defined( 'WC_VERSION' ) ) {
			return false;
		}
		return version_compare( WC_VERSION, '6.0', '>=' );
	}

	/**
	 * Display admin notices for health check issues
	 *
	 * @return void
	 */
	public static function display_admin_notices(): void {
		if ( ! is_admin() || ! current_user_can( 'manage_woocommerce' )
		) {
			return;
		}

		$health_status = self::check_health();
		if ( ! empty( $health_status['messages'] ) ) {
			foreach ( $health_status['messages'] as $message_data ) {
				$class  = isset( $message_data['type'] ) ? 'notice-' . $message_data['type'] : 'notice-error';
				$class .= isset( $message_data['dismissible'] ) && $message_data['dismissible'] ? ' is-dismissible' : '';
				echo '<div class="notice ' . esc_attr( $class ) . '"><p>' . wp_kses_post( $message_data['message'] ) . '</p></div>';
			}
		}
	}

	/**
	 * Format a message for display
	 *
	 * @param string $message The message to format.
	 * @return string The formatted message.
	 */
	private static function format_message( string $message ): string {
		return sprintf( 'AllPays: %s', $message );
	}
}
