<?php
/**
 * API Class
 *
 * @package AllPays
 * @since 1.0.0
 * @author AllPays
 * @copyright 2025 AllPays
 * @license GPL-2.0-or-later
 * @link https://allpays.co
 * @see https://allpays.co
 */

namespace AllPays\Core;

defined( 'ABSPATH' ) || exit;

use WC_Order;
use AllPays\Core\Logger;
/**
 * API Class
 */
class API {

	/**
	 * Cache key for API health status
	 *
	 * @var string
	 */
	private static string $health_cache_key = 'allpaysco_api_health';

	/**
	 * Cache key for gateways
	 *
	 * @var string
	 */
	private static string $gateways_cache_key = 'allpaysco_gateways';


	/**
	 * Cache expiration time in seconds
	 *
	 * @var int
	 */
	private static int $cache_expiration = 3600; // 1 hour

	/**
	 * Make an API request
	 *
	 * @param string $endpoint API endpoint.
	 * @param array  $args Additional wp_remote_get arguments.
	 * @return array|\WP_Error Response data or WP_Error
	 */
	private static function make_request( string $endpoint, array $args = array() ) {
		$default_args = array(
			'timeout'   => 15,
			'sslverify' => true,
			'headers'   => array(
				'Accept-Language' => strtok( get_locale(), '_' ),
			),
		);

		$args = wp_parse_args( $args, $default_args );
		$url  = rtrim( ALLPAYSCO_API_URL, '/' ) . '/' . ltrim( $endpoint, '/' );

		Logger::debug(
			'API Request: ' . $endpoint,
			array(
				'endpoint' => $endpoint,
				'args'     => $args,
			)
		);

		$response = wp_remote_get( $url, $args );

		// Log WP_Error.
		if ( is_wp_error( $response ) ) {
			Logger::error(
				sprintf( 'API request failed: %s', $response->get_error_message() ),
				array(
					'endpoint' => $endpoint,
					'args'     => $args,
					'error'    => $response->get_error_message(),
				)
			);
			return $response;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$body          = json_decode( wp_remote_retrieve_body( $response ), true );

		// Log non-200 responses.
		if ( 200 !== $response_code ) {
			Logger::error(
				sprintf( 'API returned %d response', $response_code ),
				array(
					'endpoint' => $endpoint,
					'args'     => $args,
					'response' => $body,
					'headers'  => wp_remote_retrieve_headers( $response ),
				)
			);
		}

		// Log API-level errors (when response is 200 but API indicates error).
		if ( 200 === $response_code && ( empty( $body['success'] ) || isset( $body['error'] ) ) ) {
			Logger::error(
				'API returned error response',
				array(
					'endpoint' => $endpoint,
					'args'     => $args,
					'response' => $body,
				)
			);
		}

		Logger::debug(
			'API Response: ' . $endpoint,
			array(
				'endpoint' => $endpoint,
				'code'     => wp_remote_retrieve_response_code( $response ),
				'body'     => json_decode( wp_remote_retrieve_body( $response ), true ),
			)
		);

		return $response;
	}

	/**
	 * Convert amount to USD
	 *
	 * @param float  $amount Amount to convert.
	 * @param string $currency Currency to convert from.
	 * @return float Converted amount.
	 * @throws \Exception If currency conversion fails.
	 */
	public static function convert_to_usd( float $amount, string $currency ): float {
		if ( 'USD' === $currency ) {
			return $amount;
		}

		$response = self::make_request(
			'convert?' . http_build_query(
				array(
					'value' => $amount,
					'from'  => strtolower( sanitize_text_field( $currency ) ),
				)
			),
			array( 'timeout' => 30 )
		);

		if ( is_wp_error( $response ) ) {
			throw new \Exception( esc_html__( 'Currency conversion failed', 'allpaysco-payment-gateway-for-woocommerce' ) );
		}

		$data = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( ! isset( $data['value_coin'] ) ) {
			throw new \Exception( esc_html__( 'Invalid currency conversion response', 'allpaysco-payment-gateway-for-woocommerce' ) );
		}

		return (float) $data['value_coin'];
	}

	/**
	 * Generate temporary wallet address
	 *
	 * @param WC_Order $order Order instance.
	 * @return array Wallet data.
	 * @throws \Exception If failed to generate temporary wallet address.
	 */
	public static function generate_temporary_wallet_address( WC_Order $order ): array {
		$wallet_address = get_option( ALLPAYSCO_SETTINGS_ID . '_polygon_wallet' );

		if ( empty( $wallet_address ) ) {
			throw new \Exception( esc_html__( 'Wallet address is not set.', 'allpaysco-payment-gateway-for-woocommerce' ) );
		}

		$wp_nonce = wp_create_nonce( 'allpaysco_callback_nonce' . $order->get_id() );

		$callback_url = add_query_arg(
			array(
				'order_id' => $order->get_id(),
				'nonce'    => $wp_nonce,
			),
			rest_url( 'allpays-gateway/v1/callback' )
		);

		$response = self::make_request(
			'wallets',
			array(
				'method'  => 'POST',
				'timeout' => 30,
				'body'    => wp_json_encode(
					array(
						'polygon_address' => $wallet_address,
						'callback_url'    => $callback_url,
					)
				),
				'headers' => array(
					'Content-Type' => 'application/json',
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			throw new \Exception( esc_html__( 'Failed to generate temporary wallet address request.', 'allpaysco-payment-gateway-for-woocommerce' ) );
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( empty( $body['success'] ) || empty( $body['data'] ) ||
			empty( $body['data']['encrypted_address'] ) || empty( $body['data']['polygon_address'] ) ) {
			throw new \Exception( esc_html__( 'Failed to generate temporary wallet address.', 'allpaysco-payment-gateway-for-woocommerce' ) );
		}

		$amount_usd = self::convert_to_usd( $order->get_total(), $order->get_currency() );
		$order->add_meta_data( 'allpaysco_encrypted_address', $body['data']['encrypted_address'], true );
		$order->add_meta_data( 'allpaysco_temporary_polygon_address', $body['data']['polygon_address'], true );
		$order->add_meta_data( 'allpaysco_callback_url', $callback_url, true );
		$order->add_meta_data( 'allpaysco_callback_nonce', $wp_nonce, true );
		$order->add_meta_data( 'allpaysco_amount_usd', $amount_usd, true );
		$order->save();

		return $body['data'];
	}

	/**
	 * REST API callback endpoint
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return \WP_REST_Response|\WP_Error Response object.
	 */
	public static function rest_api_callback_endpoint_v1( $request ) {
		$order_id = absint( $request->get_param( 'order_id' ) );

		if ( empty( $order_id ) ) {
			return new \WP_Error(
				'invalid_order_id',
				esc_html__( 'Invalid order ID.', 'allpaysco-payment-gateway-for-woocommerce' ),
				array( 'status' => 400 )
			);
		}

		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return new \WP_Error(
				'invalid_order',
				esc_html__( 'Invalid order.', 'allpaysco-payment-gateway-for-woocommerce' ),
				array( 'status' => 404 )
			);
		}

		$nonce = sanitize_text_field( $request->get_param( 'nonce' ) );

		if ( empty( $nonce ) || $order->get_meta( 'allpaysco_callback_nonce' ) !== $nonce ) {
			return new \WP_Error(
				'invalid_nonce',
				esc_html__( 'Invalid nonce.', 'allpaysco-payment-gateway-for-woocommerce' ),
				array( 'status' => 401 )
			);
		}

		if ( $order->get_status() !== 'on-hold' ) {
			return new \WP_Error(
				'invalid_order_status',
				esc_html__( 'Order is not on hold.', 'allpaysco-payment-gateway-for-woocommerce' ),
				array( 'status' => 400 )
			);
		}

		if ( 0 !== strpos( $order->get_payment_method(), 'allpaysco' ) ) {
			return new \WP_Error(
				'invalid_payment_method',
				esc_html__( 'Order is not paid with AllPays.', 'allpaysco-payment-gateway-for-woocommerce' ),
				array( 'status' => 400 )
			);
		}

		$txid = sanitize_text_field( $request->get_param( 'txid_out' ) );

		if ( empty( $txid ) ) {
			return new \WP_Error(
				'missing_txid',
				esc_html__( 'TXID is required.', 'allpaysco-payment-gateway-for-woocommerce' ),
				array( 'status' => 400 )
			);
		}

		$order->payment_complete( $txid );

		/* translators: 1: Transaction ID */
		$order->add_order_note( sprintf( esc_html__( 'AllPays payment completed with TXID: %s', 'allpaysco-payment-gateway-for-woocommerce' ), $txid ) );

		return new \WP_REST_Response(
			array(
				'message'  => esc_html__( 'Order payment completed.', 'allpaysco-payment-gateway-for-woocommerce' ),
				'order_id' => $order_id,
				'txid'     => $txid,
			),
			200
		);
	}

	/**
	 * Get API health status
	 *
	 * @return array|null API response data or null on error
	 */
	public static function get_health_status(): ?array {
		$response = self::make_request( 'health' );

		if ( ! is_wp_error( $response ) ) {
			$body          = json_decode( wp_remote_retrieve_body( $response ), true );
			$response_code = wp_remote_retrieve_response_code( $response );

			if ( 200 === $response_code && ! empty( $body ) ) {
				// Get cached data only for last_update comparison.
				$cached = get_transient( self::$health_cache_key );

				// Purge all caches if:
				// 1. Health check fails.
				// 2. Last update timestamp has changed.
				if ( empty( $body['success'] ) ||
					( $cached && ! empty( $cached['data']['last_update'] ) &&
					! empty( $body['data']['last_update'] ) &&
					$body['data']['last_update'] !== $cached['data']['last_update'] )
				) {
					self::purge_cache();
				}

				set_transient( self::$health_cache_key, $body, self::$cache_expiration );
				return $body;
			}
		}

		// On any error or invalid response, purge all caches.
		self::purge_cache();
		delete_transient( self::$health_cache_key );
		return null;
	}

	/**
	 * Get available payment gateways
	 *
	 * @param bool $force_check Force a fresh check bypassing cache.
	 * @return array|null Array containing payment methods and providers, or null on error
	 */
	public static function get_gateways( bool $force_check = false ): ?array {

		if ( ! $force_check ) {
			$cached = get_transient( self::$gateways_cache_key );
			if ( false !== $cached ) {
				return $cached;
			}
		}

		$response = self::make_request( 'gateways' );

		if ( ! is_wp_error( $response ) ) {
			$body = json_decode( wp_remote_retrieve_body( $response ), true );

			if ( 200 === wp_remote_retrieve_response_code( $response ) &&
			! empty( $body['success'] ) &&
			! empty( $body['data'] ) &&
			isset( $body['data']['payment_methods'] ) &&
			isset( $body['data']['payment_providers'] ) ) {
				set_transient( self::$gateways_cache_key, $body['data'], self::$cache_expiration );
				return $body['data'];
			}
		}

		delete_transient( self::$gateways_cache_key );
		return null;
	}

	/**
	 * Get available payment gateways for a given order.
	 *
	 * @param WC_Order $order         The WooCommerce order to get gateways for.
	 * @param string   $mode          The mode to get gateways for.
	 * @param bool     $force_check   Whether to force a fresh check bypassing cache.
	 * @return array   Array of available gateways or empty array on error.
	 */
	public static function get_available_gateways( WC_Order $order, string $mode = 'methods', bool $force_check = false ): array {
		// Generate a unique cache key based on order parameters.
		$cache_key = sprintf(
			'allpaysco_available_gateways_%s_%s_%s_%s',
			$order->get_total(),
			$order->get_currency(),
			$order->get_billing_country(),
			$order->get_billing_state(),
			$mode
		);

		// Try to get cached gateways if not forcing a check.
		if ( ! $force_check ) {
			$cached = get_transient( $cache_key );
			if ( false !== $cached ) {
				return $cached;
			}
		}

		// Prepare the request data based on the order.
		$request_data = array(
			'order' => array(
				'amount'   => $order->get_total(),
				'currency' => $order->get_currency(),
				'country'  => $order->get_billing_country(),
				'state'    => $order->get_billing_state(),
			),
			'mode'  => $mode,
		);

		// Make the API request using our standardized make_request method.
		$response = self::make_request(
			'gateways/available',
			array(
				'method'  => 'POST',
				'body'    => wp_json_encode( $request_data ),
				'headers' => array(
					'Content-Type' => 'application/json',
				),
			)
		);

		// Handle errors in the response.
		if ( ! is_wp_error( $response ) ) {

			$body          = json_decode( wp_remote_retrieve_body( $response ), true );
			$response_code = wp_remote_retrieve_response_code( $response );

			// Validate response and cache if successful.
			if ( 200 === $response_code &&
				true === $body['success'] &&
				! empty( $body['data'] ) ) {

				// Cache the gateway data with the unique key.
				set_transient(
					$cache_key,
					$body['data'],
					self::$cache_expiration
				);

				return $body['data'];

			}
		}

		// Clear cache and return empty array on failure.
		delete_transient( $cache_key );
		return array();
	}

	/**
	 * Purge all API caches
	 *
	 * This function clears all cached API responses including:
	 * - Health status cache
	 * - Gateways cache
	 * - Available gateways cache (using a wildcard search)
	 *
	 * @return void
	 */
	public static function purge_cache(): void {
		global $wpdb;

		// Delete specific caches.
		delete_transient( self::$health_cache_key );
		delete_transient( self::$gateways_cache_key );

		// Delete all available gateways caches using SQL wildcard.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Intentional direct query for cache cleanup
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
				'_transient_allpaysco_available_gateways_%'
			)
		);

		// Also delete timeout entries.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Intentional direct query for cache cleanup
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
				'_transient_timeout_allpaysco_available_gateways_%'
			)
		);
	}
}
