<?php
/**
 * Plugin Name: AllPays.co - Payment Gateway for WooCommerce
 * Plugin URI: https://allpays.co/woocommerce-plugin
 * Description: Accept any payment method instantly - from credit cards, Apple Pay, Google Pay, Venmo, Revolut, Robinhood, bank transfers, without KYB or registration with instant crypto settlements in one simple integration.
 * Version: 1.0.0
 * Author: AllPays.co
 * Author URI: https://allpays.co
 * Tags: payment gateway, woocommerce, traditional payments, crypto, instant payments
 * Text Domain: allpaysco-payment-gateway-for-woocommerce
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * WC requires at least: 6.0
 * WC tested up to: 9.8.5
 * Requires Plugins: woocommerce
 * License: GNU General Public License v2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @package AllPays
 */

namespace AllPays;

defined( 'ABSPATH' ) || exit;

define( 'ALLPAYSCO_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'ALLPAYSCO_PLUGIN_FILE', plugin_basename( __FILE__ ) );
define( 'ALLPAYSCO_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'ALLPAYSCO_VERSION', '1.0.0' );
define( 'ALLPAYSCO_API_URL', 'https://api.allpays.co/v1' );
define( 'ALLPAYSCO_PAYMENT_URL', 'https://api.allpays.co/v1' );
define( 'ALLPAYSCO_SETTINGS_ID', 'allpaysco_settings' );

// Autoload classes.
require_once __DIR__ . '/vendor/autoload.php';

use AllPays\Core\Registry;
use AllPays\Core\Settings;
use AllPays\Core\API;
use AllPays\Core\Translations;
use AllPays\Payment_Methods\Base_Payment_Method_Gateway;
use AllPays\Payment_Methods\Base_Payment_Provider_Gateway;
use AllPays\Abstracts\Payment_Provider;
use AllPays\Core\Utils;
/**
 * Main AllPays Payments plugin class.
 *
 * @package AllPays
 * @since 1.0.0
 */
class AllPays_Payments {
	/**
	 * Registry instance.
	 *
	 * @var \AllPays\Core\Registry
	 */
	private static $registry = null;

	/**
	 * Plugin bootstrapping.
	 */
	public static function init() {

		// Initialize the registry.
		add_filter( 'before_woocommerce_init', array( __CLASS__, 'initialize' ) );

		// Declare WooCommerce compatibility.
		add_action( 'before_woocommerce_init', array( __CLASS__, 'declare_compatibility' ) );

		// Initialize settings.
		add_action( 'woocommerce_get_settings_pages', array( __CLASS__, 'initialize_settings' ) );

		// Add settings link to plugin page.
		add_filter( 'plugin_action_links_' . ALLPAYSCO_PLUGIN_FILE, array( __CLASS__, 'add_plugin_page_settings_link' ) );

		// Register REST API callback endpoint.
		add_action( 'rest_api_init', array( __CLASS__, 'register_rest_api_callback_endpoint' ) );

		// Load translations.
		load_plugin_textdomain( 'allpaysco-payment-gateway-for-woocommerce', false, basename( __DIR__ ) . '/languages' );

		// Display admin notices.
		add_action( 'admin_notices', array( 'AllPays\Core\Health_Check', 'display_admin_notices' ) );

		// Add admin menu.
		add_action( 'admin_menu', array( __CLASS__, 'add_admin_menu' ) );
	}

	/**
	 * Get the registry instance.
	 *
	 * @return \AllPays\Core\Registry
	 */
	public static function registry() {
		return self::$registry;
	}

	/**
	 * Initialize the plugin.
	 */
	public static function initialize() {
		$is_enabled = get_option( ALLPAYSCO_SETTINGS_ID . '_enabled' );

		if ( 'yes' !== $is_enabled ) {
			return;
		}

		// Initialize registry.
		self::$registry = new Registry();

		// Get gateway mode from settings.
		$mode = get_option( ALLPAYSCO_SETTINGS_ID . '_mode', 'methods' );

		// Get all gateways from API.
		$gateways = API::get_gateways();

		if ( ! empty( $gateways ) ) {

			// Register providers.
			if ( ! empty( $gateways['payment_providers'] ) ) {
				foreach ( $gateways['payment_providers'] as $provider ) {
					$provider = new Payment_Provider( $provider['id'], $provider['name'], $provider['description'], $provider['notes'], $provider['logo'] );
					self::$registry->register_provider( $provider );
				}
			}

			if ( 'methods' === $mode && ! empty( $gateways['payment_methods'] ) ) {
				foreach ( $gateways['payment_methods'] as $method ) {
					$payment_method = new Base_Payment_Method_Gateway( $method['id'], $method['name'], $method['description'], $method['logo'] );

					foreach ( $method['supported_providers'] as $provider ) {
						$provider = self::$registry->get_provider( $provider['id'] );
						if ( $provider ) {
							$payment_method->add_provider( $provider );
						}
					}

					self::$registry->register_payment_gateway( $payment_method );
				}
			} elseif ( 'providers' === $mode && ! empty( $gateways['payment_providers'] ) ) {
				foreach ( $gateways['payment_providers'] as $provider ) {
					$providers_enabled = get_option( ALLPAYSCO_SETTINGS_ID . '_providers_enabled', array() );
					if ( ! isset( $providers_enabled[ $provider['id'] ] ) ) {
						continue;
					}

					$payment_provider = new Base_Payment_Provider_Gateway( $provider['id'], $provider['name'], $provider['description'], $provider['logo'], $provider['notes'] );
					$payment_provider->add_provider( self::$registry->get_provider( $provider['id'] ) );
					self::$registry->register_payment_gateway( $payment_provider );
				}
			}
		}

		// Initialize cart fees hook.
		Utils::init_cart_fees();

		// Initialize translations.
		Translations::init();
	}

	/**
	 * Initialize settings
	 *
	 * @param array $settings Existing settings.
	 * @return array
	 */
	public static function initialize_settings( $settings ): array {
		$settings[] = new Settings();
		return $settings;
	}

	/**
	 * Add settings link to plugin page.
	 *
	 * @param array $links Existing links.
	 * @return array
	 */
	public static function add_plugin_page_settings_link( $links ) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			admin_url( 'admin.php?page=wc-settings&tab=checkout&section=' . ALLPAYSCO_SETTINGS_ID ),
			__( 'Settings', 'allpaysco-payment-gateway-for-woocommerce' )
		);
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Declare WooCommerce compatibility
	 *
	 * @return void
	 */
	public static function declare_compatibility(): void {
		if ( ! class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
			return;
		}

		// Declare compatibility with custom order tables.
		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
			'custom_order_tables',
			__FILE__,
			true
		);

		// Declare compatibility with cart and checkout blocks.
		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
			'cart_checkout_blocks',
			__FILE__,
			true
		);
	}

	/**
	 * Register REST API callback endpoint.
	 *
	 * @return void
	 */
	public static function register_rest_api_callback_endpoint(): void {
		register_rest_route(
			'allpays-gateway/v1',
			'/callback',
			array(
				'methods'             => 'GET',
				'callback'            => array( API::class, 'rest_api_callback_endpoint_v1' ),
				'permission_callback' => '__return_true',
			)
		);
	}

	/**
	 * Add admin menu
	 */
	public static function add_admin_menu() {
		// The icon in Base64 format.
		$icon_base64 = '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';

		// The icon in the data URI scheme.
		$icon_data_uri = 'data:image/svg+xml;base64,' . $icon_base64;

		add_menu_page(
			'AllPays.co Settings',
			'AllPays.co',
			'manage_options',
			'admin.php?page=wc-settings&tab=checkout&section=' . ALLPAYSCO_SETTINGS_ID,
			'',
			$icon_data_uri,
			57
		);
	}
}

AllPays_Payments::init();
